// WebSocket connection state
let ws = null;
let isConnected = false;
let reconnectAttempts = 0;
const MAX_RECONNECT_ATTEMPTS = 5;
const RECONNECT_DELAY = 5000;

// Track subscribed wallets
let subscribedWallets = new Set();

// Track which tabs are focused
let focusedTabId = null;
let windowFocused = true;

// Store full tx hashes for notification clicks (partial hash -> full hash)
const pendingTxHashes = new Map();

// Initialize on install/startup
chrome.runtime.onInstalled.addListener(() => {
  console.log('Moggi Wallet Tracker installed');
});

// Track window focus
chrome.windows.onFocusChanged.addListener((windowId) => {
  windowFocused = windowId !== chrome.windows.WINDOW_ID_NONE;
});

// Track tab focus
chrome.tabs.onActivated.addListener((activeInfo) => {
  focusedTabId = activeInfo.tabId;
});

// Handle notification clicks
chrome.notifications.onClicked.addListener(async (notificationId) => {
  // Handle trade notifications: "moggi-trade-{tokenAddress}"
  if (notificationId.startsWith('moggi-trade-')) {
    const tokenAddress = notificationId.replace('moggi-trade-', '');
    const url = `https://gmgn.ai/monad/token/${tokenAddress}`;

    // Open or focus the URL
    const tabs = await chrome.tabs.query({ url: 'https://gmgn.ai/*' });
    if (tabs.length > 0) {
      await chrome.tabs.update(tabs[0].id, { url, active: true });
      await chrome.windows.update(tabs[0].windowId, { focused: true });
    } else {
      await chrome.tabs.create({ url });
    }

    // Clear the notification
    chrome.notifications.clear(notificationId);
  }
  // Handle transaction notifications: "moggi-tx-{txHash}"
  else if (notificationId.startsWith('moggi-tx-')) {
    // Get full tx hash from storage (we only stored partial in notificationId)
    const partialHash = notificationId.replace('moggi-tx-', '');
    // Open moggi.tools explorer - we stored partial hash, need to find full one
    // For now, just open with partial and let user see it
    // Actually, let's store full hash in a temp map
    const fullHash = pendingTxHashes.get(partialHash);
    if (fullHash) {
      const url = `https://mainnet.moggi.tools/tx/${fullHash}`;
      await chrome.tabs.create({ url });
      pendingTxHashes.delete(partialHash);
    }

    // Clear the notification
    chrome.notifications.clear(notificationId);
  }
});

chrome.runtime.onStartup.addListener(() => {
  autoConnect();
});

// Auto-connect if API key exists
async function autoConnect() {
  const result = await chrome.storage.local.get(['apiKey', 'autoConnect']);
  if (result.apiKey && result.autoConnect !== false) {
    connectWebSocket();
  }
}

// Connect to WebSocket
async function connectWebSocket() {
  if (ws && ws.readyState === WebSocket.OPEN) {
    console.log('Already connected');
    return;
  }

  const result = await chrome.storage.local.get(['apiKey', 'trackedWallets']);
  const apiKey = result.apiKey;

  if (!apiKey) {
    console.error('No API key configured');
    return;
  }

  try {
    ws = new WebSocket(`wss://api.moggi.tools/ws?key=${apiKey}`);

    ws.onopen = async () => {
      console.log('WebSocket connected');
      isConnected = true;
      reconnectAttempts = 0;
      subscribedWallets.clear();

      // Subscribe to all tracked wallets
      const wallets = result.trackedWallets || [];
      for (const wallet of wallets) {
        subscribeToWallet(wallet.address);
      }

      // Broadcast status to popup
      broadcastStatus(true);
    };

    ws.onmessage = async (event) => {
      try {
        const message = JSON.parse(event.data);
        console.log('WebSocket message:', message);

        // Handle subscription confirmations
        if (message.status === 'subscribed' || message.status === 'unsubscribed') {
          console.log(`${message.status} ${message.type}: ${message.address || ''}`);
          return;
        }

        // Handle new_transaction events (includes trade detection)
        if (message.type === 'new_transaction') {
          await handleTransactionEvent(message);
        }
      } catch (e) {
        console.error('Error parsing message:', e);
      }
    };

    ws.onerror = (error) => {
      console.error('WebSocket error:', error);
    };

    ws.onclose = () => {
      console.log('WebSocket disconnected');
      isConnected = false;
      ws = null;
      subscribedWallets.clear();
      broadcastStatus(false);

      // Auto-reconnect
      if (reconnectAttempts < MAX_RECONNECT_ATTEMPTS) {
        reconnectAttempts++;
        console.log(`Reconnecting in ${RECONNECT_DELAY}ms (attempt ${reconnectAttempts}/${MAX_RECONNECT_ATTEMPTS})`);
        setTimeout(connectWebSocket, RECONNECT_DELAY);
      }
    };
  } catch (e) {
    console.error('Failed to connect:', e);
  }
}

// Subscribe to a wallet address
function subscribeToWallet(address) {
  if (!ws || ws.readyState !== WebSocket.OPEN) return;
  if (subscribedWallets.has(address.toLowerCase())) return;

  ws.send(JSON.stringify({
    action: 'subscribe',
    type: 'address',
    address: address.toLowerCase()
  }));

  subscribedWallets.add(address.toLowerCase());
  console.log(`Subscribed to wallet: ${address}`);
}

// Unsubscribe from a wallet address
function unsubscribeFromWallet(address) {
  if (!ws || ws.readyState !== WebSocket.OPEN) return;
  if (!subscribedWallets.has(address.toLowerCase())) return;

  ws.send(JSON.stringify({
    action: 'unsubscribe',
    type: 'address',
    address: address.toLowerCase()
  }));

  subscribedWallets.delete(address.toLowerCase());
  console.log(`Unsubscribed from wallet: ${address}`);
}

// Update subscriptions when wallets change
async function updateSubscriptions(newWallets) {
  if (!ws || ws.readyState !== WebSocket.OPEN) return;

  const newAddresses = new Set(newWallets.map(w => w.address.toLowerCase()));

  // Unsubscribe from removed wallets
  for (const addr of subscribedWallets) {
    if (!newAddresses.has(addr)) {
      unsubscribeFromWallet(addr);
    }
  }

  // Subscribe to new wallets
  for (const wallet of newWallets) {
    if (!subscribedWallets.has(wallet.address.toLowerCase())) {
      subscribeToWallet(wallet.address);
    }
  }
}

// Disconnect from WebSocket
function disconnectWebSocket() {
  reconnectAttempts = MAX_RECONNECT_ATTEMPTS; // Prevent auto-reconnect
  if (ws) {
    ws.close();
    ws = null;
  }
  isConnected = false;
  subscribedWallets.clear();
  broadcastStatus(false);
}

// Check if active tab would show in-page notifications
async function isInPageNotificationVisible(showOnAllSites) {
  try {
    const [activeTab] = await chrome.tabs.query({ active: true, currentWindow: true });
    if (!activeTab?.url) return false;

    const hostname = new URL(activeTab.url).hostname.toLowerCase();

    // If showOnAllSites is enabled, in-page notifications show everywhere
    if (showOnAllSites) return true;

    // Otherwise, only show on gmgn.ai
    return hostname.includes('gmgn.ai');
  } catch (e) {
    return false;
  }
}

// Handle incoming transaction events
async function handleTransactionEvent(message) {
  const result = await chrome.storage.local.get(['trackedWallets', 'showOnAllSites', 'browserNotifications', 'showAllTransactions']);
  const trackedWallets = result.trackedWallets || [];
  const browserNotificationsEnabled = result.browserNotifications !== false;
  const showAllTransactions = result.showAllTransactions !== false; // Default true
  const showOnAllSites = result.showOnAllSites || false;

  const txData = message.data;
  const trade = message.trade; // May be null if not a trade

  // Find the tracked wallet involved in this transaction
  const fromWallet = trackedWallets.find(
    w => w.address.toLowerCase() === txData.from_address?.toLowerCase()
  );
  const toWallet = trackedWallets.find(
    w => w.address.toLowerCase() === txData.to_address?.toLowerCase()
  );

  const trackedWallet = fromWallet || toWallet;
  if (!trackedWallet) return; // Not from a tracked wallet

  const nickname = trackedWallet.nickname || shortenAddress(trackedWallet.address);

  // Check if we should show Chrome notification
  // Don't show if window is focused AND in-page notifications are visible on active tab
  const inPageVisible = windowFocused && await isInPageNotificationVisible(showOnAllSites);
  const shouldShowChromeNotification = !inPageVisible && browserNotificationsEnabled;

  // Handle detected trades (Nad.fun, bonding curves, etc.)
  if (message.transaction_type === 'trade' && trade) {
    const tradeData = {
      transaction_hash: txData.hash,
      block_number: txData.block_number,
      trader: trackedWallet.address,
      token_address: trade.token_address,
      token_symbol: trade.token_symbol,
      token_name: trade.token_name,
      token_image: trade.token_image,
      is_buy: trade.is_buy,
      mon_amount_formatted: trade.mon_amount_formatted,
      token_amount_formatted: trade.token_amount_formatted,
      nickname,
      isTracked: true,
      isTrade: true
    };

    // Build notification data
    const notificationData = {
      type: 'TRADE_NOTIFICATION',
      trade: tradeData,
      showOnAllSites: result.showOnAllSites || false
    };

    // Send to all tabs
    await broadcastToTabs(notificationData);

    // Show Chrome notification if window not focused
    if (shouldShowChromeNotification) {
      showChromeNotification(tradeData, nickname);
    }
  }
  // Handle other transactions (transfers, contract interactions, etc.)
  else if (showAllTransactions) {
    // Parse value from hex to MON
    const valueWei = BigInt(txData.value || '0');
    const valueMon = Number(valueWei) / 1e18;
    const valueFormatted = formatNumber(valueMon.toString());

    // Determine transaction type for display
    let txType = 'transfer';
    let displayType = 'TRANSFER';

    if (txData.input && txData.input.length > 10) {
      // Has calldata - likely a contract interaction
      txType = 'contract';
      displayType = 'CONTRACT';
    }

    if (valueMon > 0 && fromWallet) {
      displayType = 'SENT';
    } else if (valueMon > 0 && toWallet) {
      displayType = 'RECEIVED';
    }

    const txNotification = {
      transaction_hash: txData.hash,
      block_number: txData.block_number,
      trader: trackedWallet.address,
      from_address: txData.from_address,
      to_address: txData.to_address,
      value_formatted: valueFormatted,
      nickname,
      isTracked: true,
      isTrade: false,
      txType,
      displayType
    };

    // Build notification data
    const notificationData = {
      type: 'TRANSACTION_NOTIFICATION',
      transaction: txNotification,
      showOnAllSites: result.showOnAllSites || false
    };

    // Send to all tabs
    await broadcastToTabs(notificationData);

    // Show Chrome notification if window not focused
    if (shouldShowChromeNotification) {
      showTransactionChromeNotification(txNotification, nickname);
    }
  }
}

// Broadcast message to all tabs
async function broadcastToTabs(message) {
  const tabs = await chrome.tabs.query({});
  for (const tab of tabs) {
    if (tab.id) {
      try {
        await chrome.tabs.sendMessage(tab.id, message);
      } catch (e) {
        // Tab might not have content script loaded
      }
    }
  }
}

// Show Chrome notification for non-trade transactions
function showTransactionChromeNotification(tx, nickname) {
  const partialHash = tx.transaction_hash.slice(0, 16);
  const notificationId = `moggi-tx-${partialHash}`;

  // Store full hash for click handler
  pendingTxHashes.set(partialHash, tx.transaction_hash);

  let title = `${nickname}`;
  let message = '';

  if (tx.displayType === 'CONTRACT') {
    title += ' - Contract Interaction';
    message = `Interacted with ${shortenAddress(tx.to_address)}`;
  } else if (tx.displayType === 'SENT') {
    title += ' - Sent MON';
    message = `Sent ${tx.value_formatted} MON to ${shortenAddress(tx.to_address)}`;
  } else if (tx.displayType === 'RECEIVED') {
    title += ' - Received MON';
    message = `Received ${tx.value_formatted} MON from ${shortenAddress(tx.from_address)}`;
  } else {
    title += ' - Transaction';
    message = `TX: ${shortenAddress(tx.transaction_hash)}`;
  }

  const options = {
    type: 'basic',
    iconUrl: 'icons/icon128.jpeg',
    title,
    message,
    priority: 1,
    requireInteraction: false,
    silent: false
  };

  chrome.notifications.create(notificationId, options, (id) => {
    setTimeout(() => {
      chrome.notifications.clear(id);
      pendingTxHashes.delete(partialHash);
    }, 8000);
  });
}

// Show Chrome browser notification
function showChromeNotification(trade, nickname) {
  const isBuy = trade.is_buy;
  const action = isBuy ? '🟢 BOUGHT' : '🔴 SOLD';
  const monAmount = formatNumber(trade.mon_amount_formatted);
  const tokenAmount = formatNumber(trade.token_amount_formatted);

  const notificationId = `moggi-trade-${trade.token_address}`;

  const options = {
    type: 'basic',
    iconUrl: trade.token_image || 'icons/icon128.png',
    title: `${nickname} ${action}`,
    message: `${tokenAmount} ${trade.token_symbol || 'tokens'} for ${monAmount} MON`,
    priority: 2,
    requireInteraction: false,
    silent: false
  };

  chrome.notifications.create(notificationId, options, (id) => {
    // Auto-clear after 10 seconds
    setTimeout(() => {
      chrome.notifications.clear(id);
    }, 10000);
  });
}

// Format number with abbreviation
function formatNumber(numStr) {
  if (!numStr) return '0';

  const num = parseFloat(numStr);
  if (isNaN(num)) return numStr;

  if (num >= 1e9) return (num / 1e9).toFixed(2) + 'B';
  if (num >= 1e6) return (num / 1e6).toFixed(2) + 'M';
  if (num >= 1e3) return (num / 1e3).toFixed(2) + 'K';
  if (num >= 1) return num.toFixed(2);
  if (num >= 0.01) return num.toFixed(4);
  return num.toFixed(6);
}

// Broadcast connection status to popup
function broadcastStatus(connected) {
  chrome.runtime.sendMessage({
    type: 'CONNECTION_STATUS',
    connected
  }).catch(() => {
    // Popup might not be open
  });
}

// Handle messages from popup
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  switch (message.type) {
    case 'CONNECT_WEBSOCKET':
      reconnectAttempts = 0;
      connectWebSocket();
      sendResponse({ success: true });
      break;

    case 'DISCONNECT_WEBSOCKET':
      disconnectWebSocket();
      sendResponse({ success: true });
      break;

    case 'GET_STATUS':
      sendResponse({ connected: isConnected, subscribedCount: subscribedWallets.size });
      break;

    case 'API_KEY_UPDATED':
      // Reconnect with new API key
      if (isConnected) {
        disconnectWebSocket();
        setTimeout(() => {
          reconnectAttempts = 0;
          connectWebSocket();
        }, 1000);
      }
      sendResponse({ success: true });
      break;

    case 'WALLETS_UPDATED':
      // Update WebSocket subscriptions
      updateSubscriptions(message.wallets || []);
      sendResponse({ success: true });
      break;

    case 'SETTINGS_UPDATED':
      // Settings stored in storage, broadcast to tabs
      chrome.tabs.query({}, (tabs) => {
        tabs.forEach(tab => {
          if (tab.id) {
            chrome.tabs.sendMessage(tab.id, {
              type: 'SETTINGS_CHANGED',
              showOnAllSites: message.showOnAllSites
            }).catch(() => {});
          }
        });
      });
      sendResponse({ success: true });
      break;
  }

  return true; // Keep message channel open for async response
});

// Helper: Shorten address
function shortenAddress(address) {
  if (!address) return 'Unknown';
  return `${address.slice(0, 6)}...${address.slice(-4)}`;
}

// Auto-connect on script load
autoConnect();
