// Moggi Wallet Tracker - Content Script
(function() {
  'use strict';

  // Configuration
  const ALLOWED_DOMAINS = ['gmgn.ai', 'www.gmgn.ai'];
  const NOTIFICATION_DURATION = 8000;
  const MAX_NOTIFICATIONS = 5;

  // State
  let showOnAllSites = false;
  let notificationContainer = null;

  // Initialize
  async function init() {
    // Load settings
    try {
      const result = await chrome.storage.local.get(['showOnAllSites']);
      showOnAllSites = result.showOnAllSites || false;
    } catch (e) {
      console.error('Failed to load settings:', e);
    }

    // Create notification container
    createNotificationContainer();

    // Listen for messages
    chrome.runtime.onMessage.addListener(handleMessage);
  }

  // Create notification container
  function createNotificationContainer() {
    if (notificationContainer) return;

    notificationContainer = document.createElement('div');
    notificationContainer.id = 'moggi-notification-container';
    document.body.appendChild(notificationContainer);
  }

  // Handle incoming messages
  function handleMessage(message, sender, sendResponse) {
    if (message.type === 'TRADE_NOTIFICATION') {
      handleTradeNotification(message);
    } else if (message.type === 'TRANSACTION_NOTIFICATION') {
      handleTransactionNotification(message);
    } else if (message.type === 'SETTINGS_CHANGED') {
      showOnAllSites = message.showOnAllSites;
    }
  }

  // Check if we should show notifications on current site
  function shouldShowNotification() {
    if (showOnAllSites) return true;

    const hostname = window.location.hostname.toLowerCase();
    return ALLOWED_DOMAINS.some(domain => hostname.includes(domain));
  }

  // Handle trade notification
  function handleTradeNotification(message) {
    if (!shouldShowNotification()) return;

    const trade = message.trade;
    showTradeNotification(trade);
  }

  // Handle transaction notification (non-trade)
  function handleTransactionNotification(message) {
    if (!shouldShowNotification()) return;

    const tx = message.transaction;
    showTransactionNotification(tx);
  }

  // Show trade notification
  function showTradeNotification(trade) {
    if (!notificationContainer) {
      createNotificationContainer();
    }

    // Limit notifications
    limitNotifications();

    const notification = document.createElement('div');
    notification.className = `moggi-notification ${trade.is_buy ? 'buy' : 'sell'}`;
    notification.dataset.tokenAddress = trade.token_address;

    const isBuy = trade.is_buy;
    const actionText = isBuy ? 'BOUGHT' : 'SOLD';
    const actionEmoji = isBuy ? '🟢' : '🔴';

    // Format amounts
    const monAmount = formatNumber(trade.mon_amount_formatted);
    const tokenAmount = formatNumber(trade.token_amount_formatted);

    notification.innerHTML = `
      <div class="moggi-notification-header">
        <div class="moggi-notification-trader">
          ${trade.kol_pfp ? `<img src="${trade.kol_pfp}" alt="" class="moggi-trader-avatar">` : ''}
          <span class="moggi-trader-name">${escapeHtml(trade.nickname)}</span>
        </div>
        <button class="moggi-notification-close">&times;</button>
      </div>
      <div class="moggi-notification-content">
        <div class="moggi-trade-action">
          <span class="moggi-action-emoji">${actionEmoji}</span>
          <span class="moggi-action-text">${actionText}</span>
        </div>
        <div class="moggi-trade-details">
          <div class="moggi-token-info">
            ${trade.token_image ? `<img src="${trade.token_image}" alt="" class="moggi-token-image">` : '<div class="moggi-token-placeholder"></div>'}
            <div class="moggi-token-text">
              <span class="moggi-token-symbol">${escapeHtml(trade.token_symbol || 'Unknown')}</span>
              <span class="moggi-token-amount">${tokenAmount}</span>
            </div>
          </div>
          <div class="moggi-mon-amount">
            <span class="moggi-mon-value">${monAmount}</span>
            <span class="moggi-mon-label">MON</span>
          </div>
        </div>
      </div>
      <div class="moggi-notification-footer">
        <span class="moggi-timestamp">${formatTime(new Date())}</span>
        ${isOnGmgn() ? '<span class="moggi-click-hint">Click to view token</span>' : ''}
      </div>
    `;

    // Add click handler for navigation on GMGN
    notification.addEventListener('click', (e) => {
      if (e.target.classList.contains('moggi-notification-close')) {
        closeNotification(notification);
        return;
      }

      if (isOnGmgn() && trade.token_address) {
        window.location.href = `https://gmgn.ai/monad/token/${trade.token_address}`;
      }
    });

    // Add close button handler
    notification.querySelector('.moggi-notification-close').addEventListener('click', (e) => {
      e.stopPropagation();
      closeNotification(notification);
    });

    // Add to container with animation (prepend so newest is at bottom with column-reverse)
    notificationContainer.prepend(notification);

    // Trigger entrance animation
    requestAnimationFrame(() => {
      notification.classList.add('show');
    });

    // Auto-remove after duration
    setTimeout(() => {
      closeNotification(notification);
    }, NOTIFICATION_DURATION);
  }

  // Show transaction notification (non-trade)
  function showTransactionNotification(tx) {
    if (!notificationContainer) {
      createNotificationContainer();
    }

    // Limit notifications
    limitNotifications();

    const notification = document.createElement('div');
    notification.className = 'moggi-notification transaction';
    notification.dataset.txHash = tx.transaction_hash;

    // Determine styling and text based on type
    let typeClass = 'contract';
    let titleText = 'New Transaction';
    let subtitleText = '';

    if (tx.displayType === 'SENT') {
      typeClass = 'sent';
      titleText = 'Sent MON';
      subtitleText = `To ${shortenAddress(tx.to_address)}`;
    } else if (tx.displayType === 'RECEIVED') {
      typeClass = 'received';
      titleText = 'Received MON';
      subtitleText = `From ${shortenAddress(tx.from_address)}`;
    } else if (tx.displayType === 'CONTRACT') {
      typeClass = 'contract';
      titleText = 'New Transaction';
      subtitleText = `Contract: ${shortenAddress(tx.to_address)}`;
    }

    notification.classList.add(typeClass);

    const hasValue = parseFloat(tx.value_formatted) > 0;

    notification.innerHTML = `
      <div class="moggi-notification-header">
        <div class="moggi-notification-trader">
          <span class="moggi-trader-name">${escapeHtml(tx.nickname)}</span>
        </div>
        <button class="moggi-notification-close">&times;</button>
      </div>
      <div class="moggi-notification-content moggi-tx-content">
        <div class="moggi-tx-info">
          <span class="moggi-tx-title">${titleText}</span>
          <span class="moggi-tx-subtitle">${subtitleText}</span>
        </div>
        ${hasValue ? `
          <div class="moggi-mon-amount">
            <span class="moggi-mon-value">${tx.value_formatted}</span>
            <span class="moggi-mon-label">MON</span>
          </div>
        ` : ''}
      </div>
      <div class="moggi-notification-footer">
        <span class="moggi-timestamp">${formatTime(new Date())}</span>
        <span class="moggi-tx-hash">${shortenAddress(tx.transaction_hash)}</span>
        <span class="moggi-click-hint">Click to view tx</span>
      </div>
    `;

    // Add click handler to open tx in explorer
    notification.addEventListener('click', (e) => {
      if (e.target.classList.contains('moggi-notification-close')) {
        closeNotification(notification);
        return;
      }

      // Open transaction in moggi.tools explorer
      if (tx.transaction_hash) {
        window.open(`https://mainnet.moggi.tools/tx/${tx.transaction_hash}`, '_blank');
      }
    });

    // Add close button handler
    notification.querySelector('.moggi-notification-close').addEventListener('click', (e) => {
      e.stopPropagation();
      closeNotification(notification);
    });

    // Add to container with animation (prepend so newest is at bottom with column-reverse)
    notificationContainer.prepend(notification);

    // Trigger entrance animation
    requestAnimationFrame(() => {
      notification.classList.add('show');
    });

    // Auto-remove after duration
    setTimeout(() => {
      closeNotification(notification);
    }, NOTIFICATION_DURATION);
  }

  // Limit number of notifications (remove oldest - last in DOM since we prepend)
  function limitNotifications() {
    const existingNotifications = notificationContainer.querySelectorAll('.moggi-notification');
    if (existingNotifications.length >= MAX_NOTIFICATIONS) {
      existingNotifications[existingNotifications.length - 1].remove();
    }
  }

  // Close notification with animation
  function closeNotification(notification) {
    if (!notification || notification.classList.contains('closing')) return;

    notification.classList.add('closing');
    notification.classList.remove('show');

    setTimeout(() => {
      notification.remove();
    }, 300);
  }

  // Check if on GMGN
  function isOnGmgn() {
    return ALLOWED_DOMAINS.some(domain =>
      window.location.hostname.toLowerCase().includes(domain)
    );
  }

  // Shorten address
  function shortenAddress(address) {
    if (!address) return 'Unknown';
    return `${address.slice(0, 6)}...${address.slice(-4)}`;
  }

  // Format number with abbreviation
  function formatNumber(numStr) {
    if (!numStr) return '0';

    const num = parseFloat(numStr);
    if (isNaN(num)) return numStr;

    if (num >= 1e9) return (num / 1e9).toFixed(2) + 'B';
    if (num >= 1e6) return (num / 1e6).toFixed(2) + 'M';
    if (num >= 1e3) return (num / 1e3).toFixed(2) + 'K';
    if (num >= 1) return num.toFixed(2);
    if (num >= 0.01) return num.toFixed(4);
    return num.toFixed(6);
  }

  // Format time
  function formatTime(date) {
    return date.toLocaleTimeString([], { hour: '2-digit', minute: '2-digit' });
  }

  // Escape HTML
  function escapeHtml(text) {
    if (!text) return '';
    const div = document.createElement('div');
    div.textContent = text;
    return div.innerHTML;
  }

  // Initialize when DOM is ready
  if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', init);
  } else {
    init();
  }
})();
