// DOM Elements
const apiKeyInput = document.getElementById('apiKey');
const saveApiKeyBtn = document.getElementById('saveApiKey');
const showOnAllSitesToggle = document.getElementById('showOnAllSites');
const browserNotificationsToggle = document.getElementById('browserNotifications');
const showAllTransactionsToggle = document.getElementById('showAllTransactions');
const walletAddressInput = document.getElementById('walletAddress');
const walletNicknameInput = document.getElementById('walletNickname');
const addWalletBtn = document.getElementById('addWallet');
const walletsList = document.getElementById('walletsList');
const connectBtn = document.getElementById('connectBtn');
const disconnectBtn = document.getElementById('disconnectBtn');
const connectionStatus = document.getElementById('connectionStatus');

// Load saved settings on popup open
document.addEventListener('DOMContentLoaded', async () => {
  await loadSettings();
  await updateConnectionStatus();
});

// Load settings from storage
async function loadSettings() {
  const result = await chrome.storage.local.get(['apiKey', 'showOnAllSites', 'browserNotifications', 'showAllTransactions', 'trackedWallets']);

  if (result.apiKey) {
    apiKeyInput.value = result.apiKey;
  }

  showOnAllSitesToggle.checked = result.showOnAllSites || false;
  // Default to true for browser notifications
  browserNotificationsToggle.checked = result.browserNotifications !== false;
  // Default to true for showing all transactions
  showAllTransactionsToggle.checked = result.showAllTransactions !== false;

  renderWalletsList(result.trackedWallets || []);
}

// Save API key
saveApiKeyBtn.addEventListener('click', async () => {
  const apiKey = apiKeyInput.value.trim();
  if (!apiKey) {
    alert('Please enter an API key');
    return;
  }

  await chrome.storage.local.set({ apiKey });

  // Notify background script
  chrome.runtime.sendMessage({ type: 'API_KEY_UPDATED', apiKey });

  showToast('API key saved!');
});

// Toggle show on all sites
showOnAllSitesToggle.addEventListener('change', async () => {
  await chrome.storage.local.set({ showOnAllSites: showOnAllSitesToggle.checked });

  // Notify all tabs
  chrome.runtime.sendMessage({
    type: 'SETTINGS_UPDATED',
    showOnAllSites: showOnAllSitesToggle.checked
  });
});

// Toggle browser notifications
browserNotificationsToggle.addEventListener('change', async () => {
  await chrome.storage.local.set({ browserNotifications: browserNotificationsToggle.checked });
});

// Toggle show all transactions
showAllTransactionsToggle.addEventListener('change', async () => {
  await chrome.storage.local.set({ showAllTransactions: showAllTransactionsToggle.checked });
});

// Add wallet
addWalletBtn.addEventListener('click', async () => {
  const address = walletAddressInput.value.trim().toLowerCase();
  const nickname = walletNicknameInput.value.trim();

  if (!address || !address.startsWith('0x') || address.length !== 42) {
    alert('Please enter a valid wallet address (0x...)');
    return;
  }

  if (!nickname) {
    alert('Please enter a nickname for this wallet');
    return;
  }

  const result = await chrome.storage.local.get(['trackedWallets']);
  const wallets = result.trackedWallets || [];

  // Check if wallet already exists
  if (wallets.some(w => w.address.toLowerCase() === address)) {
    alert('This wallet is already being tracked');
    return;
  }

  wallets.push({ address, nickname });
  await chrome.storage.local.set({ trackedWallets: wallets });

  // Notify background script
  chrome.runtime.sendMessage({ type: 'WALLETS_UPDATED', wallets });

  renderWalletsList(wallets);

  // Clear inputs
  walletAddressInput.value = '';
  walletNicknameInput.value = '';

  showToast('Wallet added!');
});

// Render wallets list
function renderWalletsList(wallets) {
  if (!wallets || wallets.length === 0) {
    walletsList.innerHTML = '<div class="empty-state">No wallets tracked yet</div>';
    return;
  }

  walletsList.innerHTML = wallets.map((wallet, index) => `
    <div class="wallet-item" data-index="${index}">
      <div class="wallet-info">
        <span class="wallet-nickname">${escapeHtml(wallet.nickname)}</span>
        <span class="wallet-address">${wallet.address}</span>
      </div>
      <button class="wallet-remove" data-index="${index}">Remove</button>
    </div>
  `).join('');

  // Add remove event listeners
  walletsList.querySelectorAll('.wallet-remove').forEach(btn => {
    btn.addEventListener('click', async (e) => {
      const index = parseInt(e.target.dataset.index);
      await removeWallet(index);
    });
  });
}

// Remove wallet
async function removeWallet(index) {
  const result = await chrome.storage.local.get(['trackedWallets']);
  const wallets = result.trackedWallets || [];

  wallets.splice(index, 1);
  await chrome.storage.local.set({ trackedWallets: wallets });

  // Notify background script
  chrome.runtime.sendMessage({ type: 'WALLETS_UPDATED', wallets });

  renderWalletsList(wallets);
  showToast('Wallet removed');
}

// Connect to WebSocket
connectBtn.addEventListener('click', async () => {
  const result = await chrome.storage.local.get(['apiKey']);

  if (!result.apiKey) {
    alert('Please save your API key first');
    return;
  }

  chrome.runtime.sendMessage({ type: 'CONNECT_WEBSOCKET' });
  showToast('Connecting...');
});

// Disconnect from WebSocket
disconnectBtn.addEventListener('click', () => {
  chrome.runtime.sendMessage({ type: 'DISCONNECT_WEBSOCKET' });
  showToast('Disconnecting...');
});

// Update connection status
async function updateConnectionStatus() {
  try {
    const response = await chrome.runtime.sendMessage({ type: 'GET_STATUS' });
    updateStatusUI(response?.connected || false, response?.subscribedCount || 0);
  } catch (e) {
    updateStatusUI(false, 0);
  }
}

function updateStatusUI(connected, subscribedCount) {
  if (connected) {
    connectionStatus.classList.add('connected');
    const countText = subscribedCount > 0 ? ` (${subscribedCount})` : '';
    connectionStatus.querySelector('.status-text').textContent = `Connected${countText}`;
    connectBtn.style.display = 'none';
    disconnectBtn.style.display = 'block';
  } else {
    connectionStatus.classList.remove('connected');
    connectionStatus.querySelector('.status-text').textContent = 'Disconnected';
    connectBtn.style.display = 'block';
    disconnectBtn.style.display = 'none';
  }
}

// Listen for status updates from background
chrome.runtime.onMessage.addListener((message, sender, sendResponse) => {
  if (message.type === 'CONNECTION_STATUS') {
    updateStatusUI(message.connected);
  }
});

// Helper: Show toast notification
function showToast(message) {
  // Simple visual feedback
  const btn = document.activeElement;
  if (btn && btn.tagName === 'BUTTON') {
    const originalText = btn.textContent;
    btn.textContent = message;
    setTimeout(() => {
      btn.textContent = originalText;
    }, 1500);
  }
}

// Helper: Escape HTML
function escapeHtml(text) {
  const div = document.createElement('div');
  div.textContent = text;
  return div.innerHTML;
}

// Poll for connection status
setInterval(updateConnectionStatus, 3000);
